// // Copyright (c) 2022 - 2023 Asadeus Studios LLC.  All rights reserved.


#include "AI/AmsAIController.h"

#include "PnmsAdvancedSystemPublicPCH.h"
#include "Components/RiderControllerComponent.h"
#include "Components/MountRiderComponent.h"

#include "Interfaces/IMountablePawnInterface.h"
#include "Interfaces/IMountRiderInterface.h"

// Basic Configuration
////////////////////////////////////////////////////////////////////////////

/**
* Setup Default Properties
*/
AAmsAIController::AAmsAIController() : Super()
{
	this->riderControllerComponent = CreateDefaultSubobject<URiderControllerComponent>("RiderControllerComponent");
	this->riderControllerComponent->SetEnableActionRpcsByDefault(true);
}

void AAmsAIController::OnPossess(APawn * aPawn)
{
	Super::OnPossess(aPawn);
	this->riderControllerComponent->SetControlledPawn(aPawn);
}

// IAdmUnifiedControllerPawn
////////////////////////////////////////////////////////////////////////////

/**
* Get the Character's Controller
* This will only work on the owning client or the server
* @return class AController *
*/
class AController * AAmsAIController::GetCharacterController_Implementation() const
{
	if(!IsValid(this->riderControllerComponent))
	{
		return nullptr;
	}
	return Cast<AController>(this->riderControllerComponent->GetOwner());
}

/**
* Get the Character that the player or bot is represented as
* @return class APawn *
*/
class APawn * AAmsAIController::GetCharacterPawn_Implementation() const
{
	if(!IsValid(this->riderControllerComponent))
	{
		return nullptr;
	}

	return this->riderControllerComponent->GetOwnedPawn();
}

/**
* Get the Mount that the character is riding, null if no mount.
* @return class APawn *
*/
class AActor * AAmsAIController::GetCharacterMount_Implementation() const
{
	if(!IsValid(this->riderControllerComponent))
	{
		return nullptr;
	}

	return this->riderControllerComponent->GetMountActor();
}

/**
* Get flag indicating that the character is mounted.
* this works for either the Rider or the Mount itself
*/
bool AAmsAIController::IsMounted_Implementation() const
{
	if(!IsValid(this->riderControllerComponent))
	{
		return false;
	}

	return this->riderControllerComponent->IsMounted();
}

/**
* Called to Prepare the Pawn or Controller for Mounting
* @param mountOrRider - The Mount or Rider of the Mount
* @param linkedActor - optional linked actor to the mount
* @version 4
*/
bool AAmsAIController::PrepareToMount_Implementation(AActor * mountOrRider, AActor * linkedActor, FMountActionResponse & response)
{
	return true;
}

/**
* Prepare the Pawn or Controller for Dismounting
* @param mountOrRider - The Mount or Rider of the Mount
* @param linkedActor - optional linked actor to the mount
* @version 4
*/
bool AAmsAIController::PrepareToDismount_Implementation(AActor * mountOrRider, FMountActionResponse & response)
{
	return true;
}

// IRiderControllerInterface
////////////////////////////////////////////////////////////////////////////

/**
* Begins the Mounting Process
* @param newMount - Actor to mount to
* @param linkedActor - Linked Actor to mount to
* @param seatId - seat to mount to
* @param mountingResponse - Status Response of the mounting if it succeeded.
* @return bool - true if the mounting operation succeeded.
*/
bool AAmsAIController::BeginMountingActor_Implementation(AActor * newMount, AActor * linkedActor, struct FMountActionResponse & mountingResponse)
{
	return this->riderControllerComponent->PerformMountActor(newMount, linkedActor, mountingResponse);
}

/**
* Begins the Mounting Process
* @param newMount - Actor to mount to
* @param linkedActor - Linked Actor to mount to
* @param seatId - seat to mount to
* @param mountingResponse - Status Response of the mounting if it succeeded.
* @return bool - true if the mounting operation succeeded.
*/
bool AAmsAIController::BeginMountingActorToSeat_Implementation(AActor * newMount, AActor * linkedActor, int32 seatId, struct FMountActionResponse & mountingResponse)
{
	return this->riderControllerComponent->PerformMountActorToSeat(newMount, linkedActor, seatId, mountingResponse);
}

/**
* Starts the dismounting action
* @param response - output status response if mounting succeeded or failed
* @return bool - true if dismount succeeded.
*/
bool AAmsAIController::BeginDismountingActor_Implementation(FMountActionResponse & response)
{
	AActor * const mountActor = this->riderControllerComponent->GetMountActor();
	return this->riderControllerComponent->PerformDismountActor(mountActor, response);
}

/**
* Begin changing seats to the specified seat ID.
* @param seatId - ID of the seat to change to
* @param mountingResponse - status Response of the seat change
* @return bool - true if the seat change was successful
*/
bool AAmsAIController::BeginChangingSeatById_Implementation(int32 seatId, struct FMountActionResponse & mountingResponse)
{
	return this->riderControllerComponent->PerformChangeSeatToIndex(seatId, mountingResponse);
}

/**
* Begin changing seats to the specified seat index.
* @param seatIndex - index of the seat to change to
* @param mountingResponse - status Response of the seat change
* @return bool - true if the seat change was successful
*/
bool AAmsAIController::BeginChangingSeatToIndex_Implementation(int32 seatIndex, struct FMountActionResponse & mountingResponse)
{
	return this->riderControllerComponent->PerformChangeSeatToIndex(seatIndex, mountingResponse);
}

/**
* Get the Rider Controller Component
*/
class URiderControllerComponent * AAmsAIController::GetRiderControllerComponent_Implementation() const
{
	return this->riderControllerComponent;
}

/**
* Called to tell the controller to possess the given Pawn
*/
bool AAmsAIController::PNMS_PossessPawn_Implementation(APawn * pawnToPossess)
{
	return this->riderControllerComponent->PossessPawn(pawnToPossess);
}

/**
* Called when a new rider is added to the current mount
*/
bool AAmsAIController::OnRiderAdded_Implementation(AActor * newRider, int32 seatId)
{
	this->riderControllerComponent->RiderAdded(newRider, seatId);
	return true;
}

/**
* Called when a rider is removed from the current mount
*/
bool AAmsAIController::OnRiderRemoved_Implementation(AActor * removedRider, int32 seatId)
{
	this->riderControllerComponent->RiderRemoved(removedRider, seatId);
	return true;
}

/**
* Called when another rider changes to a new seat on the mount
*/
bool AAmsAIController::OnOtherRiderChangedSeats_Implementation(AActor * otherRider, int32 newSeatId, int32 oldSeatId)
{
	this->riderControllerComponent->OtherRiderChangedSeat(otherRider, newSeatId, oldSeatId);
	return true;
}

/**
* Called when a Mounting action has failed from the mount or rider process
* @param pnmsMountingActionResultArgs - Structure contains information relevant to teh failure of the mounting action.
*/
bool AAmsAIController::OnPnmsMountingActionFailed_Implementation(FPnmsActionEventArgs const & pnmsActionResultArgs)
{
	this->riderControllerComponent->ResetToDismounted(pnmsActionResultArgs);

	return true;
}

/**
* Called when a Mounting action has failed from the mount or rider process
* @param pnmsMountingActionResultArgs - Structure contains information relevant to the failure of the dismounting action.
*/
bool AAmsAIController::OnPnmsDisountingActionFailed_Implementation(FPnmsActionEventArgs const & pnmsActionResultArgs)
{
	this->riderControllerComponent->ResetToMounted(pnmsActionResultArgs);

	return true;
}

/**
* Called when a Mounting action has failed from the mount or rider process
* @param pnmsMountingActionResultArgs - Structure contains information relevant to the failure of the dismounting action.
*/
bool AAmsAIController::OnPnmsChangeSeatActionFailed_Implementation(FPnmsSeatChangeActionEventArgs const & pnmsSeatChangeActionResultArgs)
{
	return true;
}