// Copyright (c) 2022 - 2023 Asadeus Studios LLC.  All rights reserved.

/**
	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
	THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE DISCLAMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS
	BE LIABLE FOR ANY DIRECT, INDIRECT, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION).  HOWEVER, CAUSED AND ON ANY THEORY OF LIABILITY, WHEATHER IN CONTRACT, STRICT
	LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
*/


#include "AI/AmsAIPlayerController.h"
#include "Player/AmsPlayerController.h"
#include "Interfaces/IAmsMountRider.h"

// Engine Includes
#include "Runtime/AIModule/Classes/Blueprint/AIBlueprintHelperLibrary.h"

// Basic Configuration
//////////////////////////////////////////////////////////////////////////////////////

/**
* Called each tick to perform updates
*/
void AAmsAIPlayerController::Tick( float deltaTime )
{
	Super::Tick(deltaTime);
}

/** Called on completing current movement request */
void AAmsAIPlayerController::OnMoveCompleted( FAIRequestID RequestID, const FPathFollowingResult & Result )
{
	Super::OnMoveCompleted(RequestID, Result);
	this->bIsMovingToLocation = false;
	this->NotifyPlayerControllerTargetReached();
}

/**
* Called when this controller possesses a pawn.
*/
void AAmsAIPlayerController::OnPossess( APawn * InPawn )
{
	Super::OnPossess(InPawn);
	this->StartAutoMovePawnToLocation();
}

/**
* Sets the owning player controller
*/
void AAmsAIPlayerController::SetOwningPlayerController( APlayerController * const inNewOwningPlayerController )
{
	this->owningPlayerController = inNewOwningPlayerController;
}

// Core
//////////////////////////////////////////////////////////////////////////////////////

/**
* Move the pawn to the specified location
*/
void AAmsAIPlayerController::StartAutoMovePawnToLocation()
{
	if (!this->HasAuthority() )
	{
		return;
	}

	APawn * controlledPawn = this->GetPawn();
	if(IsValid(controlledPawn) && IAmsMountRiderInterface::ImplementsInterface(controlledPawn))
	{
		IAmsMountRiderInterface::Execute_SetAutoMoveEnabled(controlledPawn, true);
	}

	this->bIsMovingToLocation = true;
	this->MoveToLocation(this->targetLocation, this->targetLocationThreshold);
}

/**
* Notify the Player Controller that the Target location has been reached
*/
void AAmsAIPlayerController::NotifyPlayerControllerTargetReached()
{
	APawn * controlledPawn = this->GetPawn();
	if(IsValid(controlledPawn) && IAmsMountRiderInterface::ImplementsInterface(controlledPawn))
	{
		IAmsMountRiderInterface::Execute_SetAutoMoveEnabled(controlledPawn, false);
	}

	auto * const playerController = Cast<AAmsPlayerController>(this->owningPlayerController);
	if ( !IsValid( playerController ) )
	{
		return;
	}

	playerController->SwitchPawnToPlayerController();
}
