// Copyright (c) 2022 - 2023 Asadeus Studios LLC.  All rights reserved.

/**
	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
	THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE DISCLAMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS
	BE LIABLE FOR ANY DIRECT, INDIRECT, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION).  HOWEVER, CAUSED AND ON ANY THEORY OF LIABILITY, WHEATHER IN CONTRACT, STRICT
	LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
*/

#include "Pawn/Mounts/PnmsAdvancedMountCharacterBase.h"

// PNMS Includes
#include "Components/MountablePawnComponent.h"

// PNMS Interface Includes
#include "Interfaces/IMountRiderInterface.h"

// Sets default values
APnmsAdvancedMountCharacterBase::APnmsAdvancedMountCharacterBase()
{
 	// Set this character to call Tick() every frame.  You can turn this off to improve performance if you don't need it.
	PrimaryActorTick.bCanEverTick = true;

	// Properties
	this->mountablePawnComponent = CreateDefaultSubobject<UMountablePawnComponent>("MountableActorComponent");
	this->mountablePawnComponent->SetIsMountable(true);
}

/**
* Called when the Pawn is possessed by a controller
* @param NewController - new AController possessing this character
*/
void APnmsAdvancedMountCharacterBase::PossessedBy(AController * NewController)
{
	Super::PossessedBy(NewController);
	if(!NewController->IsPlayerController())
	{
		this->MountAiController = NewController;
	}
}

// IAdmUnifiedControllerPawn
///////////////////////////////////////////////////////////////////////////////////

/** 
* Get the Character's Controller
* This will only work on the owning client or the server
* @return class AController *
*/
class AController * APnmsAdvancedMountCharacterBase::GetCharacterController_Implementation() const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return nullptr;
	}

	return this->GetController();
}

/** 
* Get the Character that the player or bot is represented as
* @return class APawn *
*/
class APawn * APnmsAdvancedMountCharacterBase::GetCharacterPawn_Implementation() const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return nullptr;
	}

	return this->mountablePawnComponent->GetDriver();
}

/**
* Get the Mount that the character is riding, null if no mount.
* @return class APawn *
*/
class AActor * APnmsAdvancedMountCharacterBase::GetCharacterMount_Implementation() const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return nullptr;
	}

	return this->mountablePawnComponent->GetOwner();
}

/** 
* Get flag indicating that the character is mounted.
* this works for either the Rider or the Mount itself
*/
bool APnmsAdvancedMountCharacterBase::IsMounted_Implementation() const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	return this->mountablePawnComponent->IsMounted();
}

/**
* Called to Prepare the Pawn or Controller for Mounting
* @param mountOrRider - The Mount or Rider of the Mount
* @param linkedActor - optional linked actor to the mount
* @version 4
*/
bool APnmsAdvancedMountCharacterBase::PrepareToMount_Implementation(AActor * mountOrRider, AActor * linkedActor, FMountActionResponse & response)
{
	return true;
}

/**
* Prepare the Pawn or Controller for Dismounting
* @param mountOrRider - The Mount or Rider of the Mount
* @param linkedActor - optional linked actor to the mount
* @version 4
*/
bool APnmsAdvancedMountCharacterBase::PrepareToDismount_Implementation(AActor * mountOrRider, FMountActionResponse & response)
{
	return true;
}

// IMountablePawnInterface
////////////////////////////////////////////////////////////////////////////

/** 
* Get the Mountable Pawn Component
*/
class UMountablePawnComponent * APnmsAdvancedMountCharacterBase::GetMountablePawnComponent_Implementation() const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return nullptr;
	}

	return this->mountablePawnComponent;
}

/**
* Get flag indicating that the Pawn Is a Mountable Actor
* Note that the IADMUnifiedControllerPawn has a similar function ICharacterMountable that should return the same value
* @return bool - true if mountable
*/
bool APnmsAdvancedMountCharacterBase::IsMountableActor_Implementation() const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	return this->mountablePawnComponent->IsMountable();
}

/**
* Get flag indicating that the Pawn can currently be mounted, e.g. it has no rider or has an open seat.
* @param newRider - rider wanting to mount
* @return bool
*/
bool APnmsAdvancedMountCharacterBase::CanMountActor_Implementation(AActor * newRider) const
{
	return ( this->GetMaxRiders() > this->GetCurrentRiderCount() ) && this->IsMountableActor();
}

/**
* Get flag indicating that the Pawn currently has any number of passengers (or a driver) currently on it
* @return bool
* @Version 6
*/
bool APnmsAdvancedMountCharacterBase::HasPassengers_Implementation() const
{
	return this->GetCurrentRiderCount() > 0;
}

/**
* Get flag indicating that the supplied pawn is allowed to mount or enter the vehicle
* @param newRider - APawn that wants to mount
* @return bool - true if allowed
*/
bool APnmsAdvancedMountCharacterBase::IsMountableByPawn_Implementation(APawn * newRider) const
{
	return true;
}

/** 
* Get flag indicating that the pawn has a current driver
*/
bool APnmsAdvancedMountCharacterBase::HasDriver_Implementation() const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	return this->mountablePawnComponent->HasDriver();
}

/**
* Get flag indicating that the Driver is ready to take control and drive the vehicle or mount
* @bool
* @Version 6
*/
bool APnmsAdvancedMountCharacterBase::IsDriverReady_Implementation() const
{
	AActor * const driver = this->GetDriver();
	if(!IsValid(driver))
	{
		return true;
	}

	if(!IMountRiderInterface::ImplementsInterface(driver))
	{
		return true;
	}

	return IMountRiderInterface::Execute_IsSeatedOnMount(driver);
}

/** 
* Flag indicating that this mount must have a driver before it can be ridden
*/
bool APnmsAdvancedMountCharacterBase::MustHaveDriver_Implementation() const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	return this->mountablePawnComponent->MustHaveDriver();
}

/** 
* Get the Driver of this Mount
* @return APawn
*/
APawn * APnmsAdvancedMountCharacterBase::GetDriver_Implementation() const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return nullptr;
	}

	return this->mountablePawnComponent->GetDriver();
}

/** 
* Determine if the given seat is the driver seat of the mount.  Must be a possessable mount to have a driver seat.
*/
bool APnmsAdvancedMountCharacterBase::IsDriverSeat_Implementation(const FSeatData & seatData) const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	return this->mountablePawnComponent->IsDriverSeat(seatData);
}

/**
* Get the maximum number of Riders the mount can support
* return int32
*/
int32 APnmsAdvancedMountCharacterBase::GetMaxRiders_Implementation() const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return 0;
	}

	return this->mountablePawnComponent->GetNumSeats();
}

/**
* Get the current number of Riders the mount has
* @return int32
*/
int32 APnmsAdvancedMountCharacterBase::GetCurrentRiderCount_Implementation() const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return 0;
	}

	return this->mountablePawnComponent->GetNumRiders();
}

/**
* Find the nearest available mounting seat on the mount relatifve to the rider and desired mounting position
* @param position - EMountingPosition position the rider desires to mount
* @param riderLocation - FVector location the rider is currently standing to mount
* @param outSeatData - FMountSeatData to mount the rider to
* @param outSeatIndex - index of the seat to attach to
* @return bool - true if a seat was found
*/
bool APnmsAdvancedMountCharacterBase::FindAvailableMountingPosition_Implementation(EMountingDirection position, FVector riderLocation, FSeatData & outSeatData, int32 & outSeatIndex) const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	return this->mountablePawnComponent->FindAvailableMountingPosition(position, riderLocation, outSeatData, outSeatIndex);
}

/**
* Get flag indicating that the seat at the specified index is occupied
* @param seatIndex - int32 index of the seat to check
* @return bool - true is the seat is occupied or the index is invalid
*/
bool APnmsAdvancedMountCharacterBase::IsSeatOccupiedAtIndex_Implementation(int32 seatIndex) const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	return this->mountablePawnComponent->IsSeatOccupiedAtIndex(seatIndex);
}

/**
* Clear the specified Seat of being occupied
* @param seatIndex - index of the seat to clear
* @return bool
*/
bool APnmsAdvancedMountCharacterBase::IsSeatOccupiedById_Implementation(int32 seatId) const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	return this->mountablePawnComponent->IsSeatOccupiedById(seatId);
}

/**
* Set the seat at the specified index as occupied by the rider
* @param seatIndex - int32 index of the seat to set occupied
* @param rider - APawn to set as the rider
* @return bool - true if operaiton succeeds
* @version 4
*/
bool APnmsAdvancedMountCharacterBase::SetSeatOccupiedAtIndex_Implementation(int32 seatIndex, APawn * rider)
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	return this->mountablePawnComponent->SetSeatOccupiedAtIndex(seatIndex, rider);
}

/**
* Set the seat by the specified id as occupied by the rider
* @param seatIndex - int32 id of the seat to set occupied
* @param rider - APawn to set as the rider
* @return bool - true if operation succeeds
*/
bool APnmsAdvancedMountCharacterBase::SetSeatOccupiedById_Implementation(int32 seatId, APawn * rider)
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	return this->mountablePawnComponent->SetSeatOccupiedById(seatId, rider);
}

/**
* Clear the specified Seat of being occupied
* @param seatIndex - index of the seat to clear
* @return bool
*/
bool APnmsAdvancedMountCharacterBase::ClearSeatAtIndex_Implementation(int32 seatIndex)
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	return this->mountablePawnComponent->ClearSeatAtIndex(seatIndex);
}

/**
* Clear the specified seat from being occupied
* @param seatIndex - int32 index of the seat to clear
* @return bool
*/
bool APnmsAdvancedMountCharacterBase::ClearSeatById_Implementation(int32 seatId)
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	return this->mountablePawnComponent->ClearSeatById(seatId);
}

/**
* Get data about the seat at the specified index
* @param index - int32 index of the seat to retrieve
* @param seatData - will hold the data of the seat in question
* @return bool - true if the seat was found
*/
bool APnmsAdvancedMountCharacterBase::GetSeatDataAtIndex_Implementation(int32 index, FSeatData & seatData) const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	return this->mountablePawnComponent->GetSeatDataAtIndex(index, seatData);
}

/**
* Get data about the seat by the specified id
* @param seatId - int32 id of the seat to retrieve
* @param seatData - will hold the data of the seat in question
* @return bool - true if the seat was found
*/
bool APnmsAdvancedMountCharacterBase::GetSeatDataById_Implementation(int32 seatId, FSeatData & seatData) const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	return this->mountablePawnComponent->GetSeatDataById(seatId, seatData);
}

/**
* Get flag indicating that a rider can mount given the desired mounting position and their current location.
* @param riderLocation - FVector
* @param desiredMountingPosition - EMountingPosition
* @return bool
*/
bool APnmsAdvancedMountCharacterBase::CanMountAtPosition_Implementation(FVector riderLocation, EMountingDirection desiredMountingPosition) const
{
	int32 outSeatIndex = -1;
	FSeatData outSeatData;

	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	return this->FindAvailableMountingPosition(desiredMountingPosition, riderLocation, outSeatData, outSeatIndex);
}

/**
* Retrieves a valid relative mounting position for the mount according to where the rider is currently standing
* This function allows for a more forgiving location than the one in the Tools Library.
* @param rider - rider actor.
* @todo - Change to GetRelativeDirectionToMount
*/
EMountingDirection APnmsAdvancedMountCharacterBase::GetRelativeMountDirection_Implementation(APawn * rider) const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return EMountingDirection::InvalidSide;
	}

	return EMountingDirection::MountAnySide;
}

/**
* Get direction to dismount from.
* @param rider - AActor
* @todo - Change to GetRelativeDirectionToDismount
* @version 4
*/
EMountingDirection APnmsAdvancedMountCharacterBase::GetRelativeDismountDirection_Implementation(APawn * rider) const
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return EMountingDirection::InvalidSide;
	}

	return EMountingDirection::MountAnySide;
}

/**
* Get the Mesh Component to mount the character to.
* @param seatId - id of the seat to retrieve the body for
* @return UMeshComponent
*/
UMeshComponent * APnmsAdvancedMountCharacterBase::GetMountBody_Implementation(int32 seatId) const
{
	return this->GetMesh();
}

/** 
* Called when a rider completes their mounting procedure signaling they have occupied a seat
* @param mountedActor - actor that is now mounted
* @param seatId - id of the seat the actor has taken
*/
bool APnmsAdvancedMountCharacterBase::OnRiderFinishedMounting_Implementation(AActor * mountedActor, int32 seatId)
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	this->mountablePawnComponent->RiderFinishedMounting(mountedActor, seatId);
	return true;
}

/**
* Called when a rider completes their dismounting procedure signaling they have freed up a seat
* @param dismountedActor - actor that has left their seat
* @param seatId - id of the seat the actor has left
*/
bool APnmsAdvancedMountCharacterBase::OnRiderFinishedDismounting_Implementation(AActor * dismountedActor, int32 seatId)
{
	ENetMode netMode = this->GetNetMode();
	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	this->mountablePawnComponent->RiderFinishedDismounting(dismountedActor, seatId);

	if(this->mountablePawnComponent->IsDriverSeatId(seatId))
	{
		this->OnMountDriverDismounted();
	}

	return true;
}

/**
* Called when a rider finished moving from one seat to another signaling that a former seat has been freed up
* @param rider - rider who changed seats
* @param newSeatId - new seat the rider is occupying
* @param oldSeatId - old seat the rider has left
*/
bool APnmsAdvancedMountCharacterBase::OnRiderFinishedChangingSeats_Implementation(AActor * rider, int32 newSeatId, int32 oldSeatId)
{
	if(!IsValid(this->mountablePawnComponent))
	{
		return false;
	}

	this->mountablePawnComponent->RiderFinishedChangingSeats(rider, newSeatId, oldSeatId);
	return true;
}

// IAmsMountableActor Interface
////////////////////////////////////////////////////////////////////////////

/**
* Get the type of Mount the implementing actor is considered.
*/
EAmsMountTypes APnmsAdvancedMountCharacterBase::GetMountType_Implementation() const
{
	return this->mountType;
}

/**
* Called when the the mount driver dismounts
*/
void APnmsAdvancedMountCharacterBase::OnMountDriverDismounted_Implementation()
{
	// if the current mount is not locally controlled a player, then have the AI Controller repossess it.
	// this will work in both Multiplayer and Single Player.
	if(this->IsLocallyControlled())
	{
		return;
	}

	// Ensure we have an AI Controller that can retake possession of our mount.
	if(!IsValid(this->MountAiController))
	{
		return;
	}

	this->MountAiController->Possess(this);
}

// Network and RPCs
////////////////////////////////////////////////////////////////////////////
/**
* Setup the replicated properties for this class
*/
void APnmsAdvancedMountCharacterBase::GetLifetimeReplicatedProps(TArray<FLifetimeProperty> & OutLifetimeProps) const
{
	Super::GetLifetimeReplicatedProps(OutLifetimeProps);

	DOREPLIFETIME(ThisClass, mountablePawnComponent);
}
