// // Copyright (c) 2022 - 2023 Asadeus Studios LLC.  All rights reserved.

#pragma once

#include "CoreMinimal.h"
#include "AIController.h"

// PNMS Headers
#include "Interfaces/IADMUnifiedControllerPawn.h"
#include "Interfaces/IRiderControllerInterface.h"

// Generated headers
#include "AmsAIController.generated.h"

/**
 * Base AI Controller for AI Based Entities that can leverage the Mounting System.
 */
UCLASS()
class PNMSADVANCEDSYSTEM_API AAmsAIController : public AAIController, public IADMUnifiedControllerPawn, public IRiderControllerInterface
{
	GENERATED_BODY()
	
public:
	// Basic Configuration
	////////////////////////////////////////////////////////////////////////////

	/**
	* Setup Default Properties
	*/
	AAmsAIController();

protected:

	/**
	* Overridable native function for when this controller is asked to possess a pawn.
	* @param InPawn The Pawn to be possessed
	*/
	virtual void OnPossess(APawn * aPawn) override;

public:

	// Properties
	////////////////////////////////////////////////////////////////////////////

protected:

	// Rider Controller Component
	UPROPERTY(VisibleAnywhere, BlueprintReadOnly, Category="AMS Controller")
	TObjectPtr<class URiderControllerComponent> riderControllerComponent;

public:

	// IAdmUnifiedControllerPawn
	////////////////////////////////////////////////////////////////////////////

	/** 
	* Get the Character's Controller
	* This will only work on the owning client or the server
	* @return class AController *
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category="IADMUnifiedControllerPawn")
	class AController * GetCharacterController() const;
	virtual class AController * GetCharacterController_Implementation() const override;

	/** 
	* Get the Character that the player or bot is represented as
	* @return class APawn *
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	class APawn * GetCharacterPawn() const;
	virtual class APawn * GetCharacterPawn_Implementation() const override;

	/**
	* Get the Mount that the character is riding, null if no mount.
	* @return class APawn *
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	class AActor * GetCharacterMount() const;
	virtual class AActor * GetCharacterMount_Implementation() const override;

	/** 
	* Get flag indicating that the character is mounted.
	* this works for either the Rider or the Mount itself
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	bool IsMounted() const;
	virtual bool IsMounted_Implementation() const override;

	/**
	* Called to Prepare the Pawn or Controller for Mounting
	* @param mountOrRider - The Mount or Rider of the Mount
	* @param linkedActor - optional linked actor to the mount
	* @version 4
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	bool PrepareToMount(AActor * mountOrRider, AActor * linkedActor, FMountActionResponse & response);
	virtual bool PrepareToMount_Implementation(AActor * mountOrRider, AActor * linkedActor, FMountActionResponse & response) override;

	/**
	* Prepare the Pawn or Controller for Dismounting
	* @param mountOrRider - The Mount or Rider of the Mount
	* @param linkedActor - optional linked actor to the mount
	* @version 4
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IADMUnifiedControllerPawn")
	bool PrepareToDismount(AActor * mountOrRider, FMountActionResponse & response);
	virtual bool PrepareToDismount_Implementation(AActor * mountOrRider, FMountActionResponse & response) override;

public:

	// IRiderControllerInterface
	////////////////////////////////////////////////////////////////////////////

	/**
	* Begins the Mounting Process
	* @param newMount - Actor to mount to
	* @param linkedActor - Linked Actor to mount to
	* @param seatId - seat to mount to
	* @param mountingResponse - Status Response of the mounting if it succeeded.
	* @return bool - true if the mounting operation succeeded.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	bool BeginMountingActor(AActor * newMount, AActor * linkedActor, struct FMountActionResponse & mountingResponse);
	virtual bool BeginMountingActor_Implementation(AActor * newMount, AActor * linkedActor, struct FMountActionResponse & mountingResponse) override;

	/**
	* Begins the Mounting Process
	* @param newMount - Actor to mount to
	* @param linkedActor - Linked Actor to mount to
	* @param seatId - seat to mount to
	* @param mountingResponse - Status Response of the mounting if it succeeded.
	* @return bool - true if the mounting operation succeeded.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	bool BeginMountingActorToSeat(AActor * newMount, AActor * linkedActor, int32 seatId, struct FMountActionResponse & mountingResponse);
	virtual bool BeginMountingActorToSeat_Implementation(AActor * newMount, AActor * linkedActor, int32 seatId, struct FMountActionResponse & mountingResponse) override;

	/** 
	* Starts the dismounting action
	* @param response - output status response if mounting succeeded or failed
	* @return bool - true if dismount succeeded.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	bool BeginDismountingActor(FMountActionResponse & response);
	virtual bool BeginDismountingActor_Implementation(FMountActionResponse & response) override;

	/**
	* Begin changing seats to the specified seat ID.
	* @param seatId - ID of the seat to change to
	* @param mountingResponse - status Response of the seat change
	* @return bool - true if the seat change was successful
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	bool BeginChangingSeatById(int32 seatId, struct FMountActionResponse & mountingResponse);
	virtual bool BeginChangingSeatById_Implementation(int32 seatId, struct FMountActionResponse & mountingResponse) override;

	/**
	* Begin changing seats to the specified seat index.
	* @param seatIndex - index of the seat to change to
	* @param mountingResponse - status Response of the seat change
	* @return bool - true if the seat change was successful
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	bool BeginChangingSeatToIndex(int32 seatIndex, struct FMountActionResponse & mountingResponse);
	virtual bool BeginChangingSeatToIndex_Implementation(int32 seatIndex, struct FMountActionResponse & mountingResponse) override;

	/** 
	* Get the Rider Controller Component
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	class URiderControllerComponent * GetRiderControllerComponent() const;
	virtual class URiderControllerComponent * GetRiderControllerComponent_Implementation() const override;

	/** 
	* Called to tell the controller to possess the given Pawn
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController", meta=(DisplayName="Possess Pawn (PNMS)"))
	bool PNMS_PossessPawn(APawn * pawnToPossess);
	virtual bool PNMS_PossessPawn_Implementation(APawn * pawnToPossess) override;

	/** 
	* Called when a new rider is added to the current mount
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	bool OnRiderAdded(AActor * newRider, int32 seatId);
	virtual bool OnRiderAdded_Implementation(AActor * newRider, int32 seatId) override;

	/**
	* Called when a rider is removed from the current mount
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	bool OnRiderRemoved(AActor * removedRider, int32 seatId);
	virtual bool OnRiderRemoved_Implementation(AActor * removedRider, int32 seatId) override;

	/**
	* Called when another rider changes to a new seat on the mount
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderController")
	bool OnOtherRiderChangedSeats(AActor * otherRider, int32 newSeatId, int32 oldSeatId);
	virtual bool OnOtherRiderChangedSeats_Implementation(AActor * otherRider, int32 newSeatId, int32 oldSeatId) override;

	/**
	* Called when a Mounting action has failed from the mount or rider process
	* @param pnmsMountingActionResultArgs - Structure contains information relevant to teh failure of the mounting action.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderControlle")
	bool OnPnmsMountingActionFailed(FPnmsActionEventArgs const & pnmsActionResultArgs);
	virtual bool OnPnmsMountingActionFailed_Implementation(FPnmsActionEventArgs const & pnmsActionResultArgs) override;

	/**
	* Called when a Mounting action has failed from the mount or rider process
	* @param pnmsMountingActionResultArgs - Structure contains information relevant to the failure of the dismounting action.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderControlle")
	bool OnPnmsDisountingActionFailed(FPnmsActionEventArgs const & pnmsActionResultArgs);
	virtual bool OnPnmsDisountingActionFailed_Implementation(FPnmsActionEventArgs const & pnmsActionResultArgs) override;

	/**
	* Called when a Mounting action has failed from the mount or rider process
	* @param pnmsMountingActionResultArgs - Structure contains information relevant to the failure of the dismounting action.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category = "IRiderControlle")
	bool OnPnmsChangeSeatActionFailed(FPnmsSeatChangeActionEventArgs const & pnmsSeatChangeActionResultArgs);
	virtual bool OnPnmsChangeSeatActionFailed_Implementation(FPnmsSeatChangeActionEventArgs const & pnmsSeatChangeActionResultArgs) override;
};
