// Copyright (c) 2022 - 2023 Asadeus Studios LLC.  All rights reserved.

/**
	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
	THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE DISCLAMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS
	BE LIABLE FOR ANY DIRECT, INDIRECT, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION).  HOWEVER, CAUSED AND ON ANY THEORY OF LIABILITY, WHEATHER IN CONTRACT, STRICT
	LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
*/

#pragma once

#include "CoreMinimal.h"
#include "AIController.h"
#include "AmsAIPlayerController.generated.h"

/**
 * Basic AI Controller for Moving the character automatically
 */
UCLASS()
class PNMSADVANCEDSYSTEM_API AAmsAIPlayerController : public AAIController
{
	GENERATED_BODY()

public:

	// Basic Configuration
	//////////////////////////////////////////////////////////////////////////////////////

	/**
	* Called each tick to perform updates
	*/
	virtual void Tick(float deltaTime) override;

	/** Called on completing current movement request */
	virtual void OnMoveCompleted( FAIRequestID RequestID, const FPathFollowingResult & Result ) override;

protected:

	/**
	* Called when this controller possesses a pawn.
	*/
	virtual void OnPossess(APawn* InPawn) override;
	
public:

	// Properties
	//////////////////////////////////////////////////////////////////////////////////////

	/**
	* Get the Player Controller this AI Controller is associated with.
	*/
	UFUNCTION(BlueprintPure, BlueprintCallable, Category="Player AI Controller")
	class APlayerController* GetOwningPlayerController() const { return this->owningPlayerController; }
	
	/**
	* Sets the owning player controller
	*/
	UFUNCTION(BlueprintCallable, Category="Player AI Controller")
	void SetOwningPlayerController(class APlayerController * const inNewOwningPlayerController);

	/**
	* Set the target location to move to
	*/
	void SetTargetLocation(FVector const & newTargetLocation) { this->targetLocation = newTargetLocation; }

protected:

	// desired target location
	UPROPERTY(Transient)
	FVector targetLocation;

	// true if the pawn is currently moving to its desired location.
	UPROPERTY(Transient)
	bool bIsMovingToLocation;

	// target location threshold
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category="Player AI Controller")
	float targetLocationThreshold = 80.0f;

private:

	UPROPERTY(Transient)
	TObjectPtr<APlayerController> owningPlayerController;

public:

	// Core
	//////////////////////////////////////////////////////////////////////////////////////

	/**
	* Move the pawn to the specified location
	*/
	UFUNCTION(BlueprintCallable, Category="Player AI Controller")
	void StartAutoMovePawnToLocation();

protected:

	/**
	* Notify the Player Controller that the Target location has been reached
	*/
	void NotifyPlayerControllerTargetReached();
};
