// // Copyright (c) 2022 - 2023 Asadeus Studios LLC.  All rights reserved.

#pragma once

#include "CoreMinimal.h"
#include "Engine/DataAsset.h"
#include "Types/AmsTypes.h"
#include "AmsMountingAnimationData.generated.h"

/**
 * Animation Data for Mounting and Dismounting Vehicles 
 */
UCLASS()
class PNMSADVANCEDSYSTEM_API UAmsMountingAnimationData : public UDataAsset
{
	GENERATED_BODY()

public:
	
	/**
	* Find the AnimMontage for mounting based on the mount type, the mounting direction
	* @param mountType - type of mount being used
	* @param mountingDirection - direction that the mounting is occurring
	* @return UAnimMontage to play the animation.
	*/
	UFUNCTION(BlueprintPure, BlueprintCallable, Category="Anim Data")
	class UAnimMontage * GetMontageForMounting(EAmsMountTypes const mountType, EMountingDirection const mountingDirection) const;

	/**
	* Find the AnimMontage for dismounting based on the mount type, the mounting direction
	* @param mountType - type of mount being used
	* @param mountingDirection - direction that the mounting is occurring
	* @return UAnimMontage to play the animation.
	*/
	UFUNCTION(BlueprintPure, BlueprintCallable, Category="Anim Data")
	class UAnimMontage * GetMontageForDismounting(EAmsMountTypes const mountType, EMountingDirection const dismountingDirection) const;

	/**
	* Find the AnimMontage for changing seats based on the starting position and the ending position
	* @param mountType - type of mount being used
	* @param startingSeatName - name of the seat to start the change from
	* @param destinationSeatName - name of the seat to end the change seat at.
	*/
	UFUNCTION(BlueprintPure, BlueprintCallable, Category="Anim Data")
	class UAnimMontage * GetMontageForChangingSeats(EAmsMountTypes const mountType, FName const & startingSeatName, FName const & destinationSeatName) const;

	/**
	* Find the AnimMontage for changing seats based on the starting position and the ending position
	* @param mountType - type of mount being used
	* @param startingSeatId - name of the seat to start the change from
	* @param destinationSeatId - name of the seat to end the change seat at.
	*/
	UFUNCTION(BlueprintPure, BlueprintCallable, Category="Anim Data")
	class UAnimMontage * GetMontageForChangingSeatsById(EAmsMountTypes const mountType, int32 const & startingSeatId, int32 const & destinationSeatId) const;
	
protected:

	// The array of mount animation data for the character
	UPROPERTY(EditDefaultsOnly, BlueprintReadOnly, Category="Anim Config", meta=(TitleProperty="MountingInfoName"))
	TArray<FAmsMountAnimationData> mountingAnimationData;

	// The array of mount animation data for the character
	UPROPERTY(EditDefaultsOnly, BlueprintReadOnly, Category="Anim Config", meta = (TitleProperty = "MountingInfoName"))
	TArray<FAmsMountAnimationData> dismountingAnimationData;

	// The array of mount animation data for the character
	UPROPERTY(EditDefaultsOnly, BlueprintReadOnly, Category="Anim Config", meta = (TitleProperty = "MountingInfoName"))
	TArray<FAmsChangeSeatAnimationData> changeSeatAnimationData;
};
