// // Copyright (c) 2022 - 2023 Asadeus Studios LLC.  All rights reserved.

#pragma once

#include "CoreMinimal.h"
#include "UObject/Interface.h"
#include "Data/AmsMountingAnimationData.h"
#include "Types/AmsTypes.h"
#include "IAmsMountRider.generated.h"

// This class does not need to be modified.
UINTERFACE(MinimalAPI)
class UAmsMountRiderInterface : public UInterface
{
	GENERATED_BODY()
};

/**
 * Interface for any actor that is considered a Rider Actor, additional functionality added based on Need of Game
 */
class PNMSADVANCEDSYSTEM_API IAmsMountRiderInterface
{
	GENERATED_BODY()

public:
	
	/**
	* Gets the Mounting Animation Data Objiect for the Rider
	* @return UAmsMountingAnimationData *
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category="IAmsMountRider")
	class UAmsMountingAnimationData * GetMountingAnimationData() const;

	/**
	* Find the AnimMontage for mounting based on the mount type, the mounting direction
	* @param mountType - type of mount being used
	* @param mountingDirection - direction that the mounting is occuring
	* @return UAnimMontage to play the animation.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category="IAmsMountRider")
	class UAnimMontage * GetMontageForMounting(EAmsMountTypes const mountType, EMountingDirection const mountingDirection) const;

	/**
	* Find the AnimMontage for dismounting based on the mount type, the mounting direction
	* @param mountType - type of mount being used
	* @param mountingDirection - direction that the mounting is occuring
	* @return UAnimMontage to play the animation.
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category="IAmsMountRider")
	class UAnimMontage * GetMontageForDismounting(EAmsMountTypes const mountType, EMountingDirection const mountingDirection) const;

	/**
	* Get the AnimMontage for changing from one seat to another.
	* @param mountType - type of mount being used
	* @param startSeatName - name of the seat the change is starting from
	* @param destinationSeatName - name of the seat to end at
	* @return UAnimMontage to play
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category="IAmsMountRider")
	class UAnimMontage * GetMontageForChangingSeatsByName(EAmsMountTypes const mountType, FName const & startingSeatName, FName const & destinationSeatName) const;

	/**
	* Get the AnimMontage for changing from one seat to another.
	* @param mountType - type of mount being used
	* @param startSeatName - name of the seat the change is starting from
	* @param destinationSeatName - name of the seat to end at
	* @return UAnimMontage to play
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category="IAmsMountRider")
	class UAnimMontage * GetMontageForChangingSeatsById(EAmsMountTypes const mountType, int32 const startingSeatId, int32 const destinationSeatId) const;

	/**
	* Enables and Disabled the auto move functionality on the rider
	*/
	UFUNCTION(BlueprintNativeEvent, BlueprintCallable, Category="IAmsMountRider")
	bool SetAutoMoveEnabled(bool isEnabled);

	IMPLEMENT_INTERFACE(UAmsMountRiderInterface);
};
